<?php

namespace Modules\Flights\Http\Controllers;

use App\Models\Customer;
use App\Models\FlightsBookingList;
use App\Models\User;
use Exception;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Modules\Payments\Entities\Payment;
use Modules\StaticData\Entities\Airport;
use Modules\StaticData\Entities\AirportTranslation;
use Modules\StaticData\Entities\CityTranslation;

class FlightsController extends Controller
{
    public $suppliers;
    public $headers;
    public $module_key = 'flights';
    public $search_session;
    public $lang;
    public $currency;
    public $country;
    public $header_session;
    public $user_id;
    public $agent;
    public $ip_address;
    public $supplier_id;

    public function __construct()
    {
        $headers = request()->header();
        $this->lang = $headers['x-language-code'][0];
        $this->user_id = request()->user_id;
        $this->country = $headers['x-country-code'][0];
        $this->currency = $headers['x-currency-code'][0];
        $this->header_session = $headers['x-session'][0] ?? null;
        $this->ip_address = $headers['X-Client-Ip'][0] ?? null;
    }

    //========================================= AutoComplete ============================
    public function auto_complete($q)
    {
        // Search for airports by name in airport_translations
        $airportQuery = AirportTranslation::where('name', 'like', '%' . $q . '%')->get();

        $airportCodeQuery = Airport::where('code', 'like', '%' . $q . '%')->get();
        // Search for cities by name in city_translations
        $cityQuery = CityTranslation::where('name', 'like', '%' . $q . '%')->get();

        $response = [];

        // Process results from airport search
        foreach ($airportQuery as $row) {
            try {
                $airport = $row->airport;
                if ($airport->code != '\\N') {
                    $response[] = [
                        "Type" => "airport",
                        "Id" => $airport->id,
                        "Code" => $airport->code,
                        "Name" => $row->name,
                        "CityName" => optional($airport->city)->name ?? '',
                        "CountryName" => optional($airport->city)->country->name ?? '',
                    ];
                }
            } catch (Exception $e) {
                // Handle any exceptions if needed
            }
        }

        foreach ($airportCodeQuery as $airport) {
            try {
                $response[] = [
                    "Type" => "airport",
                    "Id" => $airport->id,
                    "Code" => $airport->code,
                    "Name" => $airport->name,
                    "CityName" => optional($airport->city)->name ?? '',
                    "CountryName" => optional($airport->city)->country->name ?? '',
                ];
            } catch (Exception $e) {
                // Handle any exceptions if needed
            }
        }

        // Process results from city search and return associated airports
        foreach ($cityQuery as $row) {
            try {
                $city = $row->city;
                $airports = $city->airports;

                foreach ($airports as $airport) {
                    if ($airport->code != '\\N') {
                        $response[] = [
                            "Type" => "airport",
                            "Id" => $airport->id,
                            "Code" => $airport->code,
                            "Name" => $airport->name,
                            "CityName" => $city->name ?? '',
                            "CountryName" => optional($city->country)->name ?? '',
                        ];
                    }
                }
            } catch (Exception $e) {
                // Handle any exceptions if needed
            }
        }
        $response = array_values(array_unique($response, SORT_REGULAR));

        return json_response(['data' => $response], 200, true, __('api.success_response'));
    }
    //========================================= Search ==================================
    public function search(Request $request)
    {
        try {
            $agent = $request->agent;
            $user_suppliers = getSupplierList($this->module_key, $agent);
            $this->suppliers = load_suppliers($user_suppliers, $this->module_key);
            $lang = $this->lang;
            $currency = $this->currency;
            $country = $this->country;
            // $nationality = getNationality($country);
            // return $nationality;
            $params = [
                'AgentId' => $agent->id,
                'LangCode' => $lang,
                'CurrencyCode' => $currency,
                // 'GuestNationality' => $nationality??'eg',
                'GuestNationality' => 'eg',
                'locale' => $lang . '-' . $country,
                'CountryCode' => $country,
                'AdultCount' => $request->adultsCount,
                'ChildCount' => $request->childrenCount,
                'InfantCount' => $request->infantsCount,
                'Type' => $request->type,
                'Cabin' => $request->cabin,
                'Legs' => $request->legs,
                'SearchSession' => $request->search_session,
                'Airline' => $request->airline ?? [],
            ];

            $search_results = [];
            $wow_sessions = [];

            $airline_list = [];
            $airport_list = [];
            $craft_list = [];

            foreach ($this->suppliers as $supplier_key => $supplier) {
                foreach ($user_suppliers as $user_supplier) {
                    if ($user_supplier['key'] === $supplier_key) {
                        $this->supplier_id = $user_supplier['id'];
                        break;
                    }
                }
                $params['SupplierKey'] = $supplier_key;
                $result = $supplier->search($params);
                $airport_list = array_merge($airport_list, $result['AirportList']);
                $airline_list = array_merge($airline_list, $result['AirlineList']);
                $craft_list = array_merge($craft_list, $result['CraftList']);

                $this->search_session = $result['WowSessionKey'];
                if ($result['Status']) {
                    $wow_sessions[] = $result['WowSessionKey'] ?? [];
                    $supplier_results = $result['Flights'];
                    $search_results = array_merge($search_results, $supplier_results);
                    usort($search_results, function ($a, $b) {
                        if ($a["Amount"] == $b["Amount"]) {
                            return 0;
                        }
                        return ($a["Amount"] < $b["Amount"]) ? -1 : 1;
                    });

                    foreach ($search_results as $k => $row) {
                        if (!in_array($row['Session']['Supplier'], ['tboair'])) {
                            foreach ($row['Options'] as $option_key => $option) {
                                foreach ($option['Legs'] as $leg_key => $leg) {
                                    foreach ($leg['Segments'] as $seg_key => $segment) {
                                        $origin_airport_code = $segment['Origin']['AirportCode'];
                                        $origin_data = isset($airports[$origin_airport_code]) ? $airports[$origin_airport_code] : '';
                                        $origin = [
                                            'AirportName' => $origin_data['Name'] ?? '',
                                            // 'CityCode' => '',
                                            'CityName' => $origin_data['City'] ?? '',
                                            'CountryCode' => $origin_data['CountryCode'] ?? "",
                                            'CountryName' => $origin_data['Country'] ?? 's',
                                        ];
                                        $search_results[$k]['Options'][$option_key]['BookingInfo']['SupplierId'] = $this->supplier_id;
                                        $search_results[$k]['Options'][$option_key]['Legs'][$leg_key]['Segments'][$seg_key]['Origin'] = array_merge($search_results[$k]['Options'][$option_key]['Legs'][$leg_key]['Segments'][$seg_key]['Origin'], $origin);

                                        $destination_airport_code = $segment['Destination']['AirportCode'];
                                        $destination_data = isset($airports[$destination_airport_code]) ? $airports[$destination_airport_code] : '';
                                        $destination = [
                                            'AirportName' => $destination_data['Name'] ?? "",
                                            // 'CityCode' => '',
                                            'CityName' => $destination_data['City'] ?? "",
                                            'CountryCode' => $destination_data['CountryCode'] ?? '',
                                            'CountryName' => $destination_data['Country'] ?? '',
                                        ];
                                        $search_results[$k]['Options'][$option_key]['Legs'][$leg_key]['Segments'][$seg_key]['Destination'] = array_merge($search_results[$k]['Options'][$option_key]['Legs'][$leg_key]['Segments'][$seg_key]['Destination'], $destination);
                                    }
                                }
                            }
                        }
                    }
                    // set_module_search($this->module_key, $this->search_session, $params, $result['data'], $params['agentId']);
                } else if (!$result['Status']) {
                    continue;
                }
            }
            $airport_list = array_values(array_unique(array_map('trim', $airport_list)));
            $airline_list = array_values(array_unique(array_map('trim', $airline_list)));
            $craft_list = array_values(array_unique(array_map('trim', $craft_list)));

            $data = [
                'status' => true,
                'message' => $search_results ? 'Flight search success' : 'No results found',
                'data' => $search_results,
                'sessions' => $wow_sessions,
                // 'search_session' => $this->search_session,
                'airport_list' => $airport_list,
                'craft_list' => $craft_list,
                'airline_list' => $airline_list,
                'airline_list' => $airline_list,
                'craft_list' => $craft_list,
            ];
            return json_response($data, 200);
        } catch (Exception $e) {
            return json_response([
                'status' => 'false',
                'data' => [],
                'message' => "Flights search error" . $e->getMessage()
            ], 200);
        }
    }

    //========================================= Select Flight ==================================
    public function select_flight(Request $request)
    {
        $input = [
            'WowSessionKey' => $this->header_session,
            'LangCode' => $this->lang,
            'CurrencyCode' => $this->currency,
            // 'GuestNationality' => getNationality($this->country),
            'GuestNationality' => 'eg',
            'locale' => $this->lang . '-' . $this->country,
            'CountryCode' => $this->country,
            'Data' => $request->data,
        ];
        $response = [];
        $wow_session = get_module_session($input['WowSessionKey']);
        if ($wow_session) {
            $input['Supplier'] = $wow_session['supplier'];
            $supplierClass = load_supplier($input['Supplier'], 'flights');
            $response = $supplierClass->select_flight($input, $wow_session);
            $response = $response['data'];
        }
        $data = [
            'data'    => isset($input['WowSessionKey']) ? $input['WowSessionKey'] : null,
            'Info'    => isset($response['Info']) ? $response['Info'] : [],
            'status'  => isset($response['Success']) ? $response['Success'] : false,
            'message' => isset($response['Message']) ? $response['Message'] : null,
        ];
        return json_response($data, 200);
    }

    //========================================= Pricing =========================================
    public function pricing(Request $request)
    {
        $agent = $request->agent;
        $input = [
            'id' => $agent->id,
            // 'UserCredits'       =>  $this->api->current_user_credits
            'WowSessionKey' => $this->header_session,
            'LangCode' => $this->lang,
            'CurrencyCode' => $this->currency,
            // 'GuestNationality' => getNationality($this->country),
            'GuestNationality' => 'eg',
            'locale' => $this->lang . '-' . $this->country,
            'CountryCode' => $this->country,
        ];
        $wow_session = get_module_session($this->header_session);
        $supplier = $wow_session['supplier'] ?? false;
        $supplier_id = $wow_session->data['BookingInfo']['SupplierId'];
        if ($supplier) {
            $input['Supplier'] = $supplier;
            $supplierClass = load_supplier($input['Supplier'], 'flights');
            $response = $supplierClass->pricing($input);
            $response['data']['SearchRequest'] = $wow_session->query;
            $response['data']['SupplierId'] = $supplier_id;
            $data = [
                'status' => $response['status'],
                'message' => $response['message'],
                'data' => $response['data']
            ];
            return json_response($data, 200);
        }
        return json_response([
            'status' => false,
            'message' => 'Error in pricing response',
        ], 200);
    }

    //========================================= Booking =========================================
    //TODO handle PAY-BOOK Routine 
    public function book(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'ContactsData.title' => 'required|in:mr,MR,MRS,MS,MISS,mrs,miss',
            'ContactsData.first_name' => 'required|string|max:255',
            'ContactsData.last_name' => 'required|string|max:255',
            'ContactsData.email' => 'required|email|max:255',
            'ContactsData.mobile.code' => 'required|string', // Adjust this based on your requirements
            'ContactsData.mobile.number' => 'required|string', // Adjust this based on your requirements
            'ContactsData.mobile.full_number' => 'required|string', // Adjust this based on your requirements
            'ContactsData.nationality' => 'required|string',
            'travellers.*.title' => 'required|in:mr,MR,MS,MRS,MISS,mrs,miss',
            'travellers.*.first_name' => 'required|string|max:255',
            'travellers.*.last_name' => 'required|string|max:255',
            'travellers.*.birthday.day' => 'required|numeric',
            'travellers.*.birthday.month' => 'required|numeric',
            'travellers.*.birthday.year' => 'required|numeric',
            'travellers.*.email' => 'required|email|max:255',
            'travellers.*.mobile.code' => 'required|string', // Adjust this based on your requirements
            'travellers.*.mobile.number' => 'required|string', // Adjust this based on your requirements
            'travellers.*.mobile.full_number' => 'required|string', // Adjust this based on your requirements
            'travellers.*.nationality' => 'required|string',
            'travellers.*.idType' => 'required|string',
            'travellers.*.idNumber' => 'required|string',
            'travellers.*.idExpiry.day' => 'required|numeric',
            'travellers.*.idExpiry.month' => 'required|numeric',
            'travellers.*.idExpiry.year' => 'required|numeric',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422); // You can choose the appropriate HTTP status code
        }

        $post_data = $request->all();
        $company = request()->agent;
        $lead_guest = $post_data['ContactsData'];
        $user = createOrUpdateUser($lead_guest, $this->module_key);
        $input = [
            'UserId' => $company->id,
            'ParentId' => $company->id,
            'WowSessionKey' => $this->header_session,
            'LangCode' => $this->lang,
            'CurrencyCode' => $this->currency,
            'GuestNationality' => $lead_guest['nationality'],
            'locale' => $this->lang . '-' . $this->country,
            'CountryCode' => $this->country,
            'Passengers' => $request->travellers,
            'LeadGuest' => $request->ContactsData,
            'IpAddress' => $this->ip_address,
            // 'Legs' => $request->Legs,
            'CustomerId' => $user->id,
            'UserCredits' => $company->balance,
        ];
        $session_data = get_module_session($input['WowSessionKey']);
        if ($session_data) {
            $session_data['Passengers'] = $input['Passengers'];
            $session_data['LeadGuest'] = $input['LeadGuest'];
            $session_data['UserId'] = $company->id;
            $supplier = $session_data['supplier'];
            // set_session('flights', $supplier, $input['WowSessionKey'], $session_data);
            $input['IsLcc'] = $session_data['Flight']['IsLcc'] ?? "";
            $input['Supplier'] = $supplier;
            $supplierClass = load_supplier($supplier, 'flights');
            try {
                $bookingResponse = $supplierClass->create_booking($input);
            } catch (Exception $e) {
                return json_response(
                    [
                        'status' => false,
                        'message' => $e->getMessage()
                    ],
                    200
                );
            }
            $available_for_pay = $bookingResponse['PaymentInfo']['Credit']['AvailableForPay'];
            if (!$available_for_pay) {
                return json_response([
                    'status' => false,
                    'message' => 'Error in Booking : No Enough Balance'
                ], 200);
            }
            if ($available_for_pay) {
                $payment = Payment::updateOrCreate(
                    ['merchant_ref' => $bookingResponse['MerchantRef']],
                    [
                        'module' => 'flights',
                        'supplier_name' => $bookingResponse['SupplierName'],
                        'net_amount' => $bookingResponse['PaymentInfo']['Price']['NetAmount'],
                        //TODO add markup
                        // 'sale_amount' => $bookingResponse['PaymentInfo']['Price']['SaleAmount'],
                        'currency' => $bookingResponse['PaymentInfo']['Price']['Currency'],
                        'status' => Payment::STATUS_UNPAID,
                        'user_id' => $bookingResponse['UserId'],
                    ]
                );
                if (!$payment) {
                    return json_response(
                        [
                            'status' => false,
                            'message' => 'Erorr in booking : Can\'t save payment info'
                        ],
                        200
                    );
                }
                $paymentId = $payment->id;
                //update payment id in booking flights list
                $bookingData = FlightsBookingList::where('id', $bookingResponse['BookingId'])->first();
                if ($bookingData) {
                    $bookingData->update([
                        'payment_id' => $paymentId,
                    ]);
                }

                $bookingResponse['user'] = $user;
                $bookingResponse['BookingData'] = $bookingData;
                $bookingResponse['merchant_ref'] = $bookingResponse['MerchantRef'];
            }

            return json_response(
                [
                    'status' => true,
                    'message' => 'Booking created successfully',
                    'data' => $bookingResponse
                ],
                200
            );
        }
        return json_response(
            [
                'status' => false,
                'messgae' => 'Error in Booking : Session Expired'

            ],
            200
        );
    }
    //======================================== Pay From Balance ==================================
    public function pay_from_balance(Request $request, $merchantReference)
    {
        $booking = get_booking_by_merchant_reference($merchantReference);
        if (!$booking) {
            return json_response([
                'status' => false,
                'message' => 'Error in Pay : can\'t find any booking record with this merchant reference'
            ], 200);
        }
        if ($booking->payment->status == Payment::STATUS_PAID) {
            return json_response([
                'status' => false,
                'message' => 'This booking is already paid before'
            ], 200);
        }

        $input = [
            'pnr' => $booking->pnr,
            'orderNum' => $booking->booking_details['orderNum'],
            'email' => $booking->booking_details['email'],
            'name' => $booking->booking_details['name'],
            'telNum' => $booking->booking_details['telNum'],
            'supplier' => $booking->booking_details['BookingNeededInfo']['Supplier'],
            'booking_details' => $booking->booking_details,
            'booking' => $booking
        ];

        $supplierClass = load_supplier($input['supplier'], 'flights');

        //Create Order
        try {
            $response = $supplierClass->create_order($input);
        } catch (Exception $e) {
            return json_response([
                'status' => false,
                'message' => 'Error in Pay , ' . $e->getMessage()
            ], 200);
        }
        $net_amount = $response['TicketResponse']->data->payAmount;
        //proccess payment
        $payment = Payment::where('id', $booking->payment_id)->first();
        $payment->update([
            'net_amount' => $net_amount
            //TODO apply markup
            // 'sale_amount' => $net_amount 
        ]);

        $user = request()->agent;
        $input = [
            'WowSessionKey' => $this->header_session,
            'LangCode' => $this->lang,
            'CurrencyCode' => $this->currency,
            // 'GuestNationality' => getNationality($this->country),
            'GuestNationality' => 'eg',
            'locale' => $this->lang . '-' . $this->country,
            'CountryCode' => $this->country,
            'UserCredits' => $user->balance,
            'UserId' => $user->id,
            'User' => $user,
            //TODO add markup
            'Amount' => $net_amount,
        ];
        $paymentResult = $this->pay($input);
        if (!$paymentResult['status']) {
            $error = ['status' => false, 'message' => 'error in Pay : Not enough credit', 'data' => []];
            return json_response($error, 200);
        }
        $payment->update([
            'status' => Payment::STATUS_PAID,
            'payment_option' => 'Balance'
        ]);

        $booking['merchant_ref'] = $merchantReference;
        return json_response(
            [
                'status' => true,
                'message' => 'Paid Successfully',
                'data' => $booking
            ],
            200
        );
    }

    private function pay($input)
    {
        $userCredits = $input['UserCredits'];
        $userNetAmount = $userCredits->credit_amount - $input['Amount'];
        if ($userNetAmount >= $userCredits->credit_allowed_amount) {
            $paymentResult = pay_from_credits($input['User'], $input['Amount']);
            if ($paymentResult == 0) {
                return [
                    'status' => true,
                    'message' => 'Payment successful',
                    'data' => $paymentResult,
                ];
            } else {
                throw new Exception('Payment failed: insufficient balance');
            }
        } else {
            throw new Exception('Payment failed: insufficient balance');
        }
    }
    //======================================= Pre Payment ====================================
    public function pre_payment($merchant_ref)
    {
        try {
            $booking = get_booking_by_merchant_reference($merchant_ref);
            if ($booking) {
                $agent = request()->agent;
                $user_credits = $agent->balance;
                $user_net_amount = $user_credits->credit_amount - $booking->amount;
                $available_for_pay = false;

                if ($user_net_amount >= $user_credits->credit_allowed_amount) {
                    $available_for_pay = true;
                }
                $currency = $booking->booking_details['BookingNeededInfo']['SearchRequest']['CurrencyCode'];
                $data['PaymentInfo'] = [
                    'Ref' => $merchant_ref,
                    'Price' => [
                        'Amount' => $booking->booking_details['BookingInfo']['Amount'],
                        'Currency' => $currency,
                    ],
                ];

                $data['PaymentInfo']['Credit'] = [
                    'AvailableForPay' => $available_for_pay,
                    'UserCredits' => $user_credits->credit_amount,
                    'CreditCurrency' => $currency,
                ];
            }
            $response = [
                'status' => $available_for_pay ? true : false,
                'message' => $available_for_pay ? 'Enough credits found' : 'You do not have enough credit',
                'data' => $data,
                // 'WowSessionKey' => $this->api->get_wow_session_key(),
            ];
            return json_response(
                $response,
                200
            );
        } catch (Exception $e) {
            return json_response(
                [
                    'status' => false,
                    'message' => 'Pre Payment request failed' . $e->getMessage(),
                    'data' => []
                ],
                500
            );
        }
    }
    //======================================= Booking Details ====================================
    public function booking_details($merchant_ref)
    {
        $booking = get_booking_by_merchant_reference($merchant_ref);
        if (!$booking) {
            return json_response([
                'status' => false,
                'message' => 'Error in Pay : can\'t find any booking record with this merchant reference'
            ], 200);
        }
        $orderNum = $booking->booking_details['orderNum'];
        $supplier = $booking->booking_details['BookingNeededInfo']['Supplier'];
        $supplierClass = load_supplier($supplier, 'flights');
        $order_details = $supplierClass->order_details($orderNum);
        return json_response([
            'status' => true,
            'message' => 'order details successful',
            'data' => $order_details
        ], 200);


        // $input = [
        //     'UserId' => $this->api->current_user_id,
        //     'WowSessionKey' => $this->api->get_wow_session_key() ?? null,
        //     'LangCode' => $this->api->get_language_code(),
        //     'CurrencyCode' => $this->api->get_currency_code(),
        //     'CountryCode' => $this->api->get_country_code(),
        //     'GuestNationality' => $this->api->get_country_by_code($this->api->get_country_code()),
        //     'locale' => $this->api->get_language_code() . '-' . $this->api->get_country_code(),
        //     // 'PNR'   =>  $pnr ?? $this->api->input->get('pnr')
        //     'MerchantRef' => $merchant_ref ?? $this->api->input->get('merchant_ref'),
        //     'UpdateDatabase' => false,
        // ];

        // if ($booking = $this->api->db
        //     ->where('booking_id', $this->api->get_merchant_ref($input['MerchantRef'])['booking_id'])
        //     ->get('flights_booking_list')
        //     ->row()
        // ) {
        //     // $input['Booking'] = $booking;
        //     // $input['orderNum'] = '916867363416082301';
        //     // $this->api->response($input);die;
        //     $supplier = $booking->supplier_name;

        //     // $input['Supplier'] = $supplier;
        //     // $input['PNR'] = $booking->pnr;
        //     // $input['SupplierSessionId'] = $booking->supplier_session_id;
        //     // $response = $this->suppliers[$supplier]->order_details($input);
        //     // $this->api->response($response);
        //     // die;
        //     // djson($response);
        //     // $response = $response['data'];
        // } else {
        //     $response['Success'] = false;
        // }

        // $this->api->response([
        //     'status' => $response['status'] ? true : false,
        //     'message' => $response['message'],
        //     'data' => $response['data'],
        // ], API_Controller::HTTP_OK);
    }

    // public function local_booking_details($merchant_ref)
    // {
    //     try {
    //         $booking = get_booking_by_merchant_reference($merchant_ref);
    //         return json_response([
    //             'status' => true,
    //             'message' => 'Booking Details Success',
    //             'data' => $booking
    //         ]);
    //     } catch (Exception $e) {
    //         return json_response([
    //             'status' => false,
    //             'message' => 'Booking Details Failed ' . $e->getMessage(),
    //             'data' => []
    //         ]);
    //     }
    // }

    public function local_booking_details_get($merchant_ref)
    {

        try {
            $bookingDetails = get_booking_by_merchant_reference($merchant_ref);;
            return $bookingDetails;
            $bookingStatus = $bookingDetails['data']['booking_status'];
            $pnr = $bookingDetails['data']['pnr'];

            // Extracting legs from the journey data
            $journeys = $bookingDetails['data']['Booking']['Response']['data']['journeys']['journey_0'];
            $legs = [];
            foreach ($journeys as $journey) {
                foreach ($journey as $leg) {
                    $legs[] = [
                        'airline' => $leg['airline'],
                        'flightNum' => $leg['flightNum'],
                        'arrival' => $leg['arrival'],
                        'arrivalDate' => $leg['arrivalDate'],
                        'arrivalTime' => $leg['arrivalTime'],
                        'departure' => $leg['departure'],
                        'departureDate' => $leg['departureDate'],
                        'departureTime' => $leg['departureTime'],
                        'bookingCode' => $leg['bookingCode'],
                    ];
                }
            }

            $response = [
                'status' => true,
                'message' => "Booking Details Success",
                'data' => [
                    'booking_status' => $bookingStatus,
                    'pnr' => $pnr,
                    'legs' => $legs,
                ],
            ];

            return json_response($response);
        } catch (Exception $e) {
            return json_response([
                'status' => false,
                'message' => 'Booking Details Failed ' . $e->getMessage(),
                'data' => [],
            ]);
        }
    }

    //======================================== Payment Details ======================================
    public function payment_details($merchant_ref)
    {
        try {
            if (!$merchant_ref) {
                return new Exception('Merchant_Ref is required');
            }
            $booking_id = explode('.', $merchant_ref)[0];
            $booking = FlightsBookingList::findOrFail($booking_id);
            $payment =
                $data['PaymentInfo'] = [
                    'Ref' =>  $merchant_ref,
                    'MerchantRef' =>  $merchant_ref,
                    // 'Supplier' =>  $booking->supplier_id,
                    'Price' => [
                        'Amount' => $booking->amount,
                        'Currency' => $booking->currency_code
                    ],
                ];
            return json_response([
                'status' => true,
                'message' => 'Payment details successfully',
                'data' => $data
            ]);
        } catch (Exception $e) {
            return json_response([
                'status' => false,
                'message' => 'Payment details failed' . $e->getMessage(),
                'data' => []
            ]);
        }
    }
}
