<?php

namespace Modules\Payments\Http\Controllers;

use App\Models\Supplier;
use Exception;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Validation\Rule;

class PaymentsController extends Controller
{
    protected $suppliers;
    protected $headers;
    protected $module_key = 'payments';
    protected $search_session;
    protected $lang;
    protected $currency;
    protected $country;
    protected $result_limit;
    protected $header_session;
    protected $ip_address;

    public function __construct()
    {
        $headers = request()->header();
        $this->lang = $headers['x-language-code'][0] ?? null;
        $this->country = $headers['x-country-code'][0] ?? null;
        $this->result_limit = $headers['result-limit'][0] ?? 0;
        $this->currency = $headers['x-currency-code'][0] ?? null;
        $this->header_session = $headers['x-session'][0] ?? null;
        $this->ip_address = $headers['x-client-ip'][0] ?? null;
    }

    public function tokenize(Request $request)
    {
        try {
            $request->validate([
                'supplier_key' => [
                    'required',
                    'exists:suppliers,key,status,1',
                    function ($attribute, $value, $fail) {
                        $matchingSupplier = Supplier::where('key', $value)
                            ->whereHas('module', function ($query) {
                                $query->where('key', 'payments')->where('status', 1);
                            })
                            ->first();

                        if (!$matchingSupplier) {
                            $fail('Supplier key is not in the list of active payment suppliers.');
                        }
                    },
                    function ($attribute, $value, $fail) use ($request) {
                        $agent = $request->agent;
                        $user_suppliers = getSupplierList($this->module_key, $agent);

                        if (empty($user_suppliers) || !collect($user_suppliers)->pluck('key')->contains($value)) {
                            $fail('Agent isn\'t subscribed to this payment suppliers');
                        }
                    },
                ],
                'merchant_ref' => ['required']
            ], [
                'supplier_key.required' => 'Supplier key is required.',
                'supplier_key.required' => 'Merchant reference is required.',
                'supplier_key.exists' => 'Supplier key is not in the list of active payment suppliers.',
            ]);

            $supplier_key = $request->supplier_key;
            $merchant_ref = $request->merchant_ref;
            $supplierClassObject = load_supplier($supplier_key, $this->module_key);
            $auth_token = $supplierClassObject->auth();
            $iframe_url = $supplierClassObject->order_register($auth_token, $merchant_ref);
            // $iframe_url = $supplierClassObject->payment_key_request($auth_token, $merchant_ref);
            return json_response([
                'status' => true,
                'message' => 'tokenization success',
                'data' => $iframe_url
            ]);
        } catch (Exception $e) {
            return json_response([
                'status' => false,
                'message' => 'Error in tokenization request : ' . $e->getMessage()
            ], 500);
        }
    }
}
