<?php

namespace Modules\Payments\Http\Controllers;

use App\Models\FlightsBookingList;
use App\Models\User;
use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Http;
use Modules\Hotels\Entities\HotelsBookingList;
use Illuminate\Support\Str;

class PaymobController extends Controller
{
    protected $api_key;
    protected $module_key = 'payments';
    protected $supplier_key = 'paymob';
    protected $settings;
    protected $expiration = 3600;
    protected $integration_id = 3335849;
    protected $auth_url = 'https://accept.paymob.com/api/auth/tokens';
    protected $order_register_url = 'https://accept.paymob.com/api/ecommerce/orders';
    protected $payment_key_request_url = 'https://accept.paymob.com/api/acceptance/payment_keys';
    protected $iframe_id = 725393; // Replace with your actual iframe ID

    public function __construct(Request $request)
    {
        $this->settings = $this->set_supplier_settings($request);
        $this->api_key = Crypt::decryptString($this->settings['api_key']);
    }

    public function set_supplier_settings(Request $request)
    {
        $agent = $request->agent;
        return getSupplierSettings($this->module_key, $this->supplier_key, $agent);
    }

    //============================================ AUTH =====================
    public function auth()
    {
        $url = $this->auth_url;
        $api_key = $this->api_key;

        $client = new Client();

        try {
            $response = $client->post($url, [
                'headers' => [
                    'Content-Type' => 'application/json',
                ],
                'json' => ['api_key' => $api_key],
                'verify' => false,
            ]);

            $data = json_decode($response->getBody(), true);
            if (isset($data['token'])) {
                return $data['token'];
            }

            throw new Exception('Authentication failed', 401);
        } catch (ClientException $e) {
            if ($e->getResponse()->getStatusCode() === 403) {
                $errorResponse = json_decode($e->getResponse()->getBody(), true);
                $errorMessage = $errorResponse['detail'] ?? 'Authentication failed';
                throw new Exception('Authentication failed: ' . $errorMessage, 403);
            }

            throw new Exception('Guzzle Error: ' . $e->getMessage(), 500);
        }
    }

    //============================================ ORDER REGISTER ============
    public function order_register($auth_token, $merchant_ref)
    {
        try {
            $merchant_ref_array = explode_merchant_ref($merchant_ref);
            $module_key = $merchant_ref_array['module_key'];
            $booking_id = $merchant_ref_array['booking_id'];
            $merchant_order_id = $merchant_ref;
            $amount_and_user_data = $this->getBookingAmountAndUserData($module_key, $booking_id);
            if ($amount_and_user_data === null) {
                throw new Exception('Booking not found or module not supported');
            }

            $amount = $amount_and_user_data['amount'];
            $amount_cents = ceil($amount * 100);
            $user_data = $amount_and_user_data['user_data'];
            // Order data
            $orderData = [
                'auth_token' => $auth_token,
                'delivery_needed' => 'false',
                'amount_cents' => $amount_cents,
                'currency' => 'EGP',
                'merchant_order_id' => $merchant_order_id . Str::random(10),
                
                'items' => [
                    [
                        'name' => $module_key . '_booking_' . $booking_id,
                        'amount_cents' => $amount_cents,
                        'name' => $module_key . '_booking_' . $booking_id,
                        'quantity' => '1',
                    ],
                ],
                'billing_data' => [
                    'first_name' => $user_data['first_name'],
                    'last_name' => $user_data['last_name'],
                    'email' => $user_data['email'],
                    'phone_number' => $user_data['phone'],
                ],
                'integration_id' => $this->integration_id,
            ];

            // Send the order registration request
            $response = Http::withoutVerifying()->post($this->order_register_url, $orderData);
            $orderResponse = $response->json();
            $billing_data = [
                "apartment" => "NA",
                "email" => $user_data['email'],
                "floor" => "NA",
                "first_name" => $user_data['first_name'],
                "street" => "NA",
                "building" => "NA",
                "phone_number" => $user_data['phone'],
                "shipping_method" => "",
                "postal_code" => "NA",
                "city" => "NA",
                "country" => "NA",
                "last_name" => $user_data['last_name'],
                "state" => "NA",
            ];
            if (!isset($orderResponse['id'])) {
                throw new Exception('dublicate or not correct info');
            }
            return $this->payment_key_request($auth_token, $amount_cents, $this->expiration, $orderResponse['id'], $billing_data, 'EGP', $this->integration_id);
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }

    private function getBookingAmountAndUserData($module_key, $booking_id)
    {
        if ($module_key === 'hotels') {
            $booking = HotelsBookingList::where('id', $booking_id)->first();
        } elseif ($module_key === 'flights') {
            $booking = FlightsBookingList::where('id', $booking_id)->first();
        } else {
            throw new Exception('Booking module is wrong');
        }
        if ($booking) {
            $amount = $booking->amount;
            $user_data = $booking->user;
            $first_name = $user_data['first_name'];
            $last_name = $user_data['last_name'];
            $email = $user_data['email'];
            $phone = $user_data['phone'];

            return [
                'amount' => $amount,
                'user_data' => [
                    'first_name' => $first_name,
                    'last_name' => $last_name,
                    'email' => $email,
                    'phone' => $phone,
                ],
            ];
        }

        return null;
    }

    //============================================ PAYMENT KEY REQUEST =============================
    public function payment_key_request($auth_token, $amount_cents, $expiration, $order_id, $billing_data, $currency, $integration_id)
    {
        try {
            $paymentKeyData = [
                'auth_token' => $auth_token,
                'amount_cents' => $amount_cents,
                'expiration' => $expiration,
                'order_id' => $order_id,
                'billing_data' => $billing_data,
                'currency' => $currency,
                'integration_id' => $integration_id,
            ];

            $response = Http::withoutVerifying()->post($this->payment_key_request_url, $paymentKeyData);

            // $response = Http::post($this->payment_key_request_url, $paymentKeyData);
            $responseData = $response->json();
            if (isset($responseData['token'])) {
                $payment_token = $responseData['token'];
                return $this->build_iframe_url($payment_token);
            } else {
                throw new Exception('Payment key request failed');
            }
        } catch (Exception $e) {
            throw new Exception('Error in payment key request: ' . $e->getMessage());
        }
    }

    //======================================= RETURN URL =================
    public function build_iframe_url($payment_token)
    {
        return "https://accept.paymobsolutions.com/api/acceptance/iframes/{$this->iframe_id}?payment_token=$payment_token";
    }
}
