<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Module;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Crypt;

class SuppliersController extends Controller
{
    /**
     * Display a listing of the users.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $page_title = __('All Suppliers');
        $resultQuery = Supplier::query();
        $resultQuery->where('is_deleted', 0)->where('status', 1);

        if ($request->filled('name')) {
            $resultQuery->where('name', 'like', "%{$request->input('name')}%");
        }
        if ($request->filled('module_key')) {
            $resultQuery->where('module_key', 'like', "%{$request->input('module_key')}%");
        }

        $sortBy = $request->get('sort') ? $request->get('sort') : 'id';
        $direction = $request->get('direction') ? $request->get('direction') : 'asc';
        $resultQuery->orderBy($sortBy, $direction);

        $suppliers = $resultQuery->paginate(config('Reading.nodes_per_page'));
        return view('admin.suppliers.index', compact('suppliers', 'page_title'));
    }

    /**
     * Show the form for creating a new user.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $modules = Module::where('status', 1)->get();
        $page_title = __('Add New Supplier');
        $screenOption = config('page.ScreenOption');
        return view('admin.suppliers.create', compact('screenOption', 'page_title', 'modules'));
    }

    /**
     * Store a newly created user in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->validate(
            $request,
            [
                'name'        => 'required',
                'key'        => 'required|unique:suppliers,key',
                'module_id' => 'required',
            ],
        );
        $supplier = [
            'name'        => $request->name,
            'key'        => $request->key,
            'module_id' => $request->module_id,
            'settings' => $this->prepare_settings($request),
            'status' => $request->status ?? 1,
            'desc'        => $request->desc ?? "",
        ];

        $supplier = Supplier::create($supplier);
        if ($supplier) {
            return redirect()->route('admin.suppliers.index')->with('success', __('Added successfully'));
        } else {
            return redirect()->back()->with('error', __('There are some problem in form submition.'));
        }
    }


    public function prepare_settings($request)
    {
        $settings = $request->settings;
        $outputArray = [];

        if (count($settings) > 0) {
            foreach ($settings as $item) {
                list($key, $value) = $item;
                $encryptedValue = Crypt::encryptString($value);
                $outputArray[$key] = $encryptedValue;
            }
            return $outputArray;
        } else {
            return [];
        }
    }


    /**
     * Show the form for editing the specified user.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $page_title = __('Edit Supplier');
        $supplier = Supplier::findorFail($id);
        $modules = Module::get();
        return view('admin.suppliers.edit', compact('supplier', 'modules', 'page_title'));
    }

    /**
     * Update the specified user in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $this->validate(
            $request,
            [
                'name' => 'required',
                'key' => [
                    'required',
                    Rule::unique('suppliers', 'key')->ignore($id),
                ],
                'module_id' => 'required',
            ],
        );

        $supplier = Supplier::findorFail($id);
        $supplier_data = [
            'name'        => $request->name,
            'key'        => $request->key,
            'module_id' => $request->module_id,
            'settings' => $this->prepare_settings($request),
            'status' => $request->status ?? 1,
            'desc'        => $request->desc ?? "",
        ];
        $supplier->update($supplier_data);

        if ($supplier->save()) {
            return redirect()->route('admin.suppliers.index')->with('success', __('Updated successfully'));
        } else {
            return redirect()->back()->with('error', __('There are some problem in form submition.'));
        }
    }

    /**
     * Remove the specified user from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $supplier = Supplier::findorFail($id);
        if ($supplier->delete()) {
            return redirect()->back()->with('success', __('Deleted successfully'));
        } else {
            return redirect()->back()->with('error', __('There are some problem.'));
        }
    }
}
