<?php

namespace Modules\Hotels\Http\Controllers;

use App\Models\User;
use Carbon\Carbon;
use Exception;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use GuzzleHttp\Client;
use GuzzleHttp\Psr7;
use GuzzleHttp\Exception\RequestException;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Modules\Hotels\Entities\Hotel;
use Modules\Markup\Markup;
use Modules\Common\Entities\Session;
use Modules\Settings\Entities\UserSettings;

use function PHPUnit\Framework\isEmpty;

class HotelsController extends Controller
{
    public $client;
    public $headers;
    public function __construct(Request $req)
    {
        $this->headers = [
            // 'X-Authorization-Token' =>  env('API_TOKEN'),
            'Authorization' => 'Bearer ' . env('API_TOKEN'), // Assuming env() function retrieves the API_TOKEN from environment variables
            'X-Language-Code' => $req->header('X-Language-Code') ?? 'ar',
            'X-Country-Code' => $req->header('X-Country-Code') ?? 'SA',
            'X-Currency-Code' => $req->header('X-Currency-Code') ?? 'SAR',
            'X-Client-Ip' => $req->ip()
        ];

        if ($req->header('X-Session')) {
            $this->headers['X-Session'] = $req->header('X-Session');
        }

        $this->client = new Client([
            'base_uri' => env('API_MODULE_BASE'),
            'headers' => $this->headers
        ]);
    }

    public function auto_complete($query)
    {
        $response = $this->client->request('GET', 'hotels/auto_complete/' . $query);
        return response((string) $response->getBody());
    }

    public function static_result(Request $req)
    {
        $response = $this->client->get('hotels/static_result', ['query' => $req->all()]);
        return response((string) $response->getBody());
    }

    public function search(Request $req)
    {
        try {
            $user_settings = getUserSettings();
            $timeout = $user_settings ? $user_settings->search_timeout : 0;
            $response = $this->client->get('hotels/search', [
                'query' => $req->all(),
                'timeout' => $timeout ?? 0, // Response timeout
                // 'connect_timeout' => 5, // Connection timeout
            ]);
            $body = json_decode($response->getBody()->getContents());
            // return $body;
            $sessions = $body->data->Sessions;
            $query = $req->all();

            foreach ($sessions as $session) {
                Session::create([
                    'module' => 'hotels',
                    'wow_session_key' => $session,
                    'query' => $query,
                    'data' => []
                ]);
            }

            $hotels = $body->data->Hotels;
            $cityIDS = array_unique(array_column($hotels, 'CityWowCode'));
            $supplierIDS = array_unique(array_column($hotels, 'SupplierKey'));
            $hotelIDS = array_unique(array_column($hotels, 'HotelWowCode'));
            $company_id = $req->get('parent');
            $user_id = $req->user()->id;
            $engine = new Markup();
            $engine->prepare([
                'modules' => ['Hotels'],
                'suppliers' => $supplierIDS,
                'hotels' => $hotelIDS,
                'cities' => $cityIDS,
                'company_id' => $company_id,
                'employee_id' => $user_id,
                'reservation_date' => $query['checkin'],
                'booking_date' => date('Y-m-d')
            ], $company_id);


            foreach ($hotels as $key => $hotel) {
                $currency = $hotel->Currency;
                $price = $hotel->Price;
                $new_price = updateCurrencyRate($price, $currency);
                $new_currency = getBaseCurrency();
                $hotels[$key]->Price = $new_price;
                $hotels[$key]->PriceOld = $new_price;
                $hotels[$key]->Currency = $new_currency;
                $hotels[$key]->Finance->Price = $new_price;
                $hotels[$key]->Finance->Currency = $new_currency;

                $hotels[$key]->Price = $engine->applyMarkups($hotel->Price, [
                    'modules' => ['Hotels'],
                    'supplier_id' => $hotel->SupplierKey,
                    'hotel_id' => $hotel->HotelWowCode,
                    'city_id' => $hotel->CityWowCode ?? ''
                ]);
            }
            $body->data->Hotels = $hotels;
            return $body;
        } catch (\Exception $e) {
            return json_response(null, 200, false, trans('Error in hotels search ' . $e->getMessage()));
        }
    }

    public function get_user_search_timeout()
    {
        $user_id = Auth::user()->id;
        if ($user_id) {
            $setting = UserSettings::where('user_id', $user_id)->first();
            return $setting ? $setting->search_timeout : null;
        }
        return null;
    }
    // public function local_hotel_details($id, Request $req)
    // {
    //     $response = $this->client->get('hotels/local_hotel_details/' . $id);
    //     return response((string) $response->getBody());
    // }
    public function local_hotel_details($id, Request $req)
    {
        $headers = [
            // 'X-Authorization-Token' =>  env('API_TOKEN'),
            'X-Access-Token' => '8765c279-2582-40ef-8721-7d03450e0e098',
            'X-Language-Code' => $req->header('X-Language-Code'),
            'X-Country-Code' => $req->header('X-Country-Code'),
            'X-Currency-Code' => $req->header('X-Currency-Code'),
            'X-Client-Ip' => $req->ip(),
        ];

        if ($req->header('X-Session')) {
            $headers['X-Session'] = $req->header('X-Session');
        }

        $this->client = new Client([
            'base_uri' => env('API_MODULE_BASE'),
            'headers' => $headers,
            'verify' => false,
        ]);

        $response = $this->client->get('https://backoffice.wow-ws.com/api/module/hotels/local_hotel_details/' . $id);

        // $response = $this->client->get('hotels/local_hotel_details/' . $id);
        return response()->json(json_decode((string) $response->getBody()));
    }


    public function available_rooms(Request $req)
    {
        $response = $this->client->get('hotels/available_rooms', ['query' => $req->all()]);
        $body = json_decode($response->getBody()->getContents());
        /** Add Markup  */
        $data = $body->data;
        $session = Session::where('wow_session_key', $data->SessionId)->first();
        $blocks = $body->data->Blocks;
        // get ids required for markup
        $cityID = $data->CityWowCode;
        $supplierID = $data->SupplierKey;
        $hotelID = $data->HotelWowCode;
        // update session data for later usage
        $session->data = [
            'cityID' => $cityID,
            'supplierID' => $supplierID,
            'hotelID' => $hotelID,
        ];
        $session->save();
        $company_id = $req->get('parent');
        $user_id = $req->user()->id;
        $engine = new Markup();
        $engine->prepare([
            'modules' => ['Hotels'],
            'suppliers' => [$supplierID],
            'hotels' => [$hotelID],
            'cities' => [$cityID],
            'company_id' => $company_id,
            'employee_id' => $user_id,
            'reservation_date' => $session->query['checkin'],
            'booking_date' => date('Y-m-d')
        ], $company_id);

        foreach ($blocks as $key => $block) {
            $currency = $blocks[$key]->BlockPrefCurrency;
            $amount = $blocks[$key]->BlockPrefPrice;
            $blocks[$key]->BlockPrefCurrency = getBaseCurrency();
            $blocks[$key]->BlockPrefPrice = updateCurrencyRate($amount, $currency);
            $blocks[$key]->BlockPrefPriceOld = $block->BlockPrefPrice;
            $blocks[$key]->BlockPrefPrice = $engine->applyMarkups($block->BlockPrefPrice, [
                'modules' => ['Hotels'],
                'supplier_id' => $supplierID,
                'hotel_id' => $hotelID,
                'city_id' => $cityID
            ]);
        }
        $body->data->Blocks = $blocks;
        return $body;
        // return response((string) $response->getBody());
    }

    public function available_rooms_by_hotel(Request $req)
    {
        $response = $this->client->get('hotels/available_rooms_by_hotel', ['query' => $req->all()]);
        return response((string) $response->getBody());
    }

    public function pricing(Request $req)
    {
        $response = $this->client->get('hotels/pricing', ['query' => $req->all()]);
        $body = json_decode($response->getBody()->getContents(), true);
        if ($body['status']) {
            /** Add Markup  */
            $session = Session::where([
                'module' => 'hotels',
                'wow_session_key' => $req->get('session_id'),
            ])->first();
            $company_id = $req->get('parent');
            $user_id = $req->user()->id;
            $engine = new Markup();
            $engine->prepare([
                'modules' => ['Hotels'],
                'suppliers' => [$session->data['supplierID']],
                'hotels' => [$session->data['hotelID']],
                'cities' => [$session->data['cityID']],
                'company_id' => $company_id,
                'employee_id' => $user_id,
                'reservation_date' => $session->query['checkin'],
                'booking_date' => date('Y-m-d')
            ], $company_id);

            //base currency price
            $base_currency = getBaseCurrency();
            $currency_base_price = updateCurrencyRate($body['data']['TotalOriginalPrefPrice'], $body['data']['PrefCurrency']);
            $body['data']['PrefCurrency'] = $base_currency;
            $body['data']['TotalOriginalPrefPrice'] = $currency_base_price;
            $body['data']['TotalPrefPrice'] = $currency_base_price;

            //apply markup
            $body['data']['TotalPrefPrice'] = $engine->applyMarkups($currency_base_price, [
                'modules' => ['Hotels'],
                'supplier_id' => $session->data['supplierID'],
                'hotel_id' => $session->data['hotelID'],
                'city_id' => $session->data['cityID']
            ]);

            //get value of markup by every hotel room
            $total_markup_value = $body['data']['TotalPrefPrice'] - $body['data']['TotalOriginalPrefPrice'];
            $room_count = count($body['data']['HotelRooms']);
            $room_markup = $total_markup_value / $room_count;

            //apply markup to each room
            foreach ($body['data']['HotelRooms'] as $roomKey => $room) {
                $room['RoomRate']['@attributes']['PrefPrice'] = updateCurrencyRate($room['RoomRate']['@attributes']['PrefPrice'], $room['RoomRate']['@attributes']['PrefCurrency']);
                $room['RoomRate']['@attributes']['PrefPrice'] += $room_markup;
                $room['RoomRate']['@attributes']['PrefCurrency'] = $base_currency;
                $body['data']['HotelRooms'][$roomKey]['RoomRate']['@attributes']['PrefPrice'] = number_format($room['RoomRate']['@attributes']['PrefPrice'], 3, '.', '');
                $body['data']['HotelRooms'][$roomKey]['RoomRate']['@attributes']['PrefCurrency'] = $base_currency;
            }

            $session->data = [
                'amount' => $body['data']['TotalPrefPrice'],
                'amount_old' => $body['data']['TotalOriginalPrefPrice'],
                'supplierID' => $session->data['supplierID'],
                'hotelID' => $session->data['hotelID'],
                'cityID' => $session->data['cityID']
            ];
            $session->save();
        }
        return $body;
    }
    /**
     * Make a hotel booking request.
     *
     * @param Request $request The request object.
     *
     * @return Response The response object.
     */
    public function booking(Request $req)
    {
        try {
            $response = $this->client->post('hotels/booking', ['json' => $req->all()]);
            $body = json_decode($response->getBody()->getContents());
            if ($body->status == true) {
                $session = get_Session_data($req);
                $data = $body->data->bookingData;
                $amount = $session->data['amount'];
                $original_amount = $session->data['amount_old'];
                $data->amount = $amount;
                $data->original_amount = $original_amount;
                $user = createUSer($body->data->user);
                $this->upsert_booking($data, $user);
                return $body;
            }
            return json_response(null, 200, false, $body->message);
        } catch (Exception $e) {
            return json_response(null, 200, false, $e->getMessage());
        }
    }

    public function getBookingDetails($merchant_ref, Request $req)
    {
        $response = $this->client->get('hotels/booking_details/' . $merchant_ref);
        $book = Hotel::where('wow_merchant_ref', $merchant_ref)->first();
        $body = json_decode($response->getBody()->getContents(), true);
        if ($book) {
            $body['data']['amount'] = $book->amount;
            $body['data']['original_amount'] = $book->original_amount;
        } else {
            $session = get_Session_data($req);
            $body['data']['amount'] = $session->data['amount'];
            $body['data']['original_amount'] = $session->data['amount_old'];
            $markup_value =  $body['data']['amount'] - $body['data']['original_amount'];
            $room_markup = $markup_value / count($body['data']['BookingDetails']['SelectedRooms']);
            foreach ($body['data']['BookingDetails']['SelectedRooms'] as $roomKey => $room) {
                $room['RoomRate']['Price'] += $room_markup;
                $body['data']['BookingDetails']['SelectedRooms'][$roomKey]['RoomRate']['Price'] = number_format($room['RoomRate']['Price'], 3, '.', '');
            }
        }
        return $body;
        // return json_decode((string) $response->getBody(), true);
    }

    public function booking_details($merchant_ref)
    {
        $book = Hotel::with('user')->where('wow_merchant_ref', $merchant_ref)->first();
        $responseData = [
            'hotel' => $book->hotel,
            'confirmation_number' => $book->confirmation_number,
            'invoice_number' => $book->invoice_number,
            'LeadGuest' => $book->booking_details['booking_needed_details']['lead_guest'],
            'BookingDetails' => [
                'SelectedRooms' => $book->booking_details['selected_rooms'],
                'RoomCount'=>count($book->booking_details['selected_rooms'])
            ],
            'RoomsGuests'=>$book->booking_details['room_guests']

        ];
        return json_response($responseData);
    }

    public function pre_payment($merchant_ref, Request $req)
    {
        $book = Hotel::where('wow_merchant_ref', $merchant_ref)->first();
        if (!$book) {
            return json_response(null, 200, false, trans('web.no_booking_data'));
        }
        if ($book && $book->booking_status != 'Vouchered') {
            $response = $this->client->get('hotels/pre_payment/' . $merchant_ref);
            $data = json_decode($response->getBody()->getContents());
            $session = get_Session_data($req);
            $amount = $session->data['amount'];
            $amount_old = $session->data['amount_old'];
            $data->data->PaymentInfo->Price->Amount = $amount;
            $data->data->PaymentInfo->Price->Currency = $book->currency_code;
            $data->data->PaymentInfo->Price->Amount_old = $amount_old;
            $data->data->PaymentInfo->Credit->UserCredits = updateCurrencyRate($data->data->PaymentInfo->Credit->UserCredits, $data->data->PaymentInfo->Credit->CreditCurrency);
            $data->data->PaymentInfo->Credit->CreditCurrency = getBaseCurrency();
            // return $req->user()->credit_amount;
            // if ($req->user()->type != 'super_admin') {
            // $data->data->PaymentInfo->Credit->UserCredits = $req->user()->credit_amount;
            // }
            return $data;
        }

        return json_response(['PaymentInfo' => [
            'Ref' => $merchant_ref,
            'Status' => 'Paid'
        ]], 200, true, 'this booking is paied');
    }

    public function payment_details($merchant_ref)
    {
        $response = $this->client->get('hotels/payment_details/' . $merchant_ref);

        return response((string) $response->getBody());
    }

    public function invoice($merchant_ref, Request $req)
    {
        $book = Hotel::where('wow_merchant_ref', $merchant_ref)->first();
        if (!$book) {
            return json_response(null, 200, false, trans('web.no_booking_data'));
        }
        if ($book && $book->booking_status != 'Vouchered') {
            // $price = get_price($req);
            $response = $this->client->post("hotels/invoice/$merchant_ref", ['json' => $req->all()]);
            $body = json_decode($response->getBody()->getContents());
            $this->upsert_booking($body->data);
            // update_balance($req, $price);
            recordPayment($req, $merchant_ref);
            return json_response([
                'status' => true,
                'message' => 'invoice success',
                'data' => $body->data->invoice_number,

            ]);
        }
        return json_response(null, 200, false, trans('web.already_booked'));
    }

    public function check_email($email, Request $req)
    {
        $response = $this->client->get("hotels/invoice/$email", ['query' => $req->all()]);
        return response((string) $response->getBody());
    }

    // private function upsert_booking($merchant_ref, Request $req)
    // {
    //     $data = $this->getBookingDetails($merchant_ref, $req)['data'];
    //     $parent_id = $req->get('parent');
    //     if ($data) {
    //         Hotel::updateOrCreate([
    //             'user_id' =>  Auth::id(),
    //             'company_id' =>  $parent_id,
    //             'wow_merchant_ref' =>  $merchant_ref,
    //         ], [
    //             'payment_status' => $data['BookingDetails']['PaymentStatus'],
    //             // 'payment_deadline' =>  $data['payment_deadline'],
    //             // 'supplier_booking_id' => $data['supplier_booking_id'],
    //             // 'supplier_name' => $data['supplier'], //
    //             // 'supplier_ref_number' => $data['supplier_ref_number'],
    //             // 'amendment_status' =>  $data['amendment_status'],
    //             'booking_status' => $data['BookingDetails']['BookingStatus'],
    //             // 'voucher_status' => $data['voucher_status'],
    //             // 'confirmation_number' => $data['confirmation_number'],
    //             // 'invoice_number' => $data['invoice_number'],
    //             // 'hotel_id' =>  $data['hotel_id'],
    //             // 'trip_name' => $data['trip_name'],
    //             'hotel_name' => $data['Hotel']['Name'],
    //             'hotel_rating' =>  $data['Hotel']['Rating'],
    //             // 'city_code' => $data['city_code'],
    //             // 'city_name' => $data['city_name'],
    //             'from_date' => $data['BookingDetails']['Checkin'],
    //             'to_date' => $data['BookingDetails']['Checkout'],
    //             // 'last_cancellation_deadline' => $data['last_cancellation_deadline'],
    //             // 'current_cancellation_charge' => $data['current_cancellation_charge'],
    //             'rooms_number' =>  count($data['BookingDetails']['SelectedRooms']),
    //             'rooms_type_name' => $data['BookingDetails']['SelectedRooms'][0]['RoomTypeName'],
    //             'currency_code' => $data['BookingDetails']['Currency'],
    //             'amount' =>  $data['amount'],
    //             'original_amount' => $data['original_amount'],
    //             // 'booking_date' => $data['booking_date'],
    //         ]);
    //     }
    // }
    private function upsert_booking($data, $user = null)
    {
        $company_id =  request()->get('parent');
        if ($data) {
            $updateData = [
                'company_id' => $company_id,
                'payment_status' => false,
                'payment_option' => 'card',
                'payment_deadline' => $data->payment_deadline,
                'supplier_booking_id' => $data->supplier_booking_id,
                'supplier_ref_number' => $data->supplier_ref_number,
                'voucher_status' => $data->voucher_status,
                'invoice_number' => $data->invoice_number,
                'trip_name' => $data->trip_name,
                'hotel_id' => $data->hotel_id,
                'last_cancellation_deadline' => $data->last_cancellation_deadline,
                'current_cancellation_charge' => $data->current_cancellation_charge,
                'rooms_number' => $data->rooms_number,
                'rooms_type_name' => $data->rooms_type_name,
                'amendment_status' => $data->amendment_status,
                'booking_status' => $data->booking_status,
                'booking_details' => $data->booking_details,
                'confirmation_number' => $data->confirmation_number,
                'hotel_name' => $data->hotel->name,
                'hotel' => $data->hotel,
                'from_date' => $data->from_date,
                'to_date' => $data->to_date,
                'booking_details' => $data->booking_details,
                'supplier_name' => $data->supplier->name,
                'amount' => $data->amount,
                'original_amount' => $data->original_amount,
                'currency_code' => getBaseCurrency(),
                'booking_date' => Carbon::now(),
            ];

            // Check if $user is not null before adding 'user_id' to the $updateData array
            if ($user) {
                $updateData['user_id'] = $user->id;
            }

            Hotel::updateOrCreate(
                [
                    'wow_merchant_ref' => implode('.', [$data->id, $data->customer_id]),
                ],
                $updateData
            );
        }
    }

    // get local booking list for normal users
    public function local_booking_list(Request $req)
    {
        $company_id =  $req->get('parent');
        $user = Auth::user();
        if ($user->type == 'super_admin') {
            $list = Hotel::with(['user', 'company'])->orderBy('id', 'DESC')->jsonPaginate();
        } elseif ($user->type == 'company') {
            $list = Hotel::with('user')->where('company_id', $company_id)->orderBy('id', 'DESC')->jsonPaginate();
        } else {
            $list = Hotel::with('user')->where('user_id', $user->id)->orderBy('id', 'DESC')->jsonPaginate();
        }

        return json_response_pagination($list);
    }

    //get remote booking list from wow backoffice for that b2b client
    public function remote_booking_list(Request $req)
    {
        $response = $this->client->post('hotels/local_booking_list/?page=' . $req->get('page') ?? 1, ['json' => $req->all()]);
        return response((string) $response->getBody());
    }

    public function cancel_booking($merchant_ref)
    {
        $response = $this->client->get('hotels/cancel_booking/' . $merchant_ref);
        $body = json_decode($response->getBody()->getContents(), true);
        if ($body['status'] == 'success') {
            Hotel::updateOrCreate([
                'wow_merchant_ref' =>  $merchant_ref,
            ], [
                'booking_status' => 'canceled',
            ]);
        }
        return json_response(null, 200, false, $body['message']);
    }

    public function country_list(Request $req)
    {
        $response = $this->client->get('hotels/country_list', ['json' => $req->all()]);
        return response()->json(json_decode((string) $response->getBody()));
    }

    public function nationality_list(Request $req)
    {
        $response = $this->client->get('hotels/nationality_list', ['json' => $req->all()]);
        return response()->json(json_decode((string) $response->getBody()));
    }
}
